;==========================================================
; MULTIDESK CARTRIDGE MENU SYSTEM v24
; Boris "docbobo" Schneider-Johne
; Find docs here: www.dreisechzig.net/multidesk
; Assembler: ACME
;==========================================================

!cpu 6502

;----------------------------------------------------------
; MEMORY MAP
;----------------------------------------------------------
; ZERO PAGE ($22-$2F Safe Zone)
zp_item_idx     = $22
zp_text_len     = $23
zp_center_off   = $24
zp_joy_state    = $25
zp_row_idx      = $2C

; 16-bit Pointers
zp_text_ptr     = $26
zp_screen_ptr   = $28
zp_color_ptr    = $2A

; RAM VARIABLES ($C000 Safe RAM)
VAR_BASE        = $C000
SelectionIndex  = VAR_BASE + 0
InputDelay      = VAR_BASE + 1
TimerLo         = VAR_BASE + 2
TimerHi         = VAR_BASE + 3
TempColor       = VAR_BASE + 4

;----------------------------------------------------------
; HARDWARE CONSTANTS
;----------------------------------------------------------
SCREEN_RAM  = $0400
COLOR_RAM   = $D800
TAPE_BUFFER = $033C
MAGIC_LATCH = $DE00

VIC_CTRL1   = $D011
VIC_BORDER  = $D020
VIC_BG      = $D021
VIC_MEM_PTR = $D018

CIA1_PRA    = $DC00
CIA1_PRB    = $DC01
CIA2_PRA    = $DD00

; KERNAL Routines
IOINIT      = $FF84
RESTOR      = $FF8A
CINT        = $FF81
SETMSG      = $FF90
GETIN       = $FFE4
SCNKEY      = $FF9F
RESET_VEC   = $FFFC

;----------------------------------------------------------
; VICE MAGIC DESK HEADER OR STRAIGHT BIN?
; Use the header option only for CRT_Linker. Read docs!
;----------------------------------------------------------

VICEHEADER = 0

!if VICEHEADER = 1 {

  !to "boot.crt", plain
  
  *=$7FB0
  !tx "C64 CARTRIDGE   "
  !byte $00, $00, $00, $40, $01, $00, $00, $13, $00, $01, $00, $00, $00, $00, $00, $00
  !tx "MULTIDESK V1"
  *=$7FE0
  !byte $00, $00, $00, $00, $00, $00, $00, $00, $00, $00, $00, $00, $00, $00, $00, $00 
  !byte $43, $48, $49, $50, $00, $00, $20, $10, $00, $00, $00, $00, $80, $00, $20, $00

  } else {      

  !to "boot.bin", plain

  }

;----------------------------------------------------------
; CARTRIDGE HEADER
;----------------------------------------------------------
* = $8000
    !word Launcher
    !word Launcher
    !byte $C3, $C2, $CD, $38, $30
    
;----------------------------------------------------------
; INCLUDE MENU DATA FROM SEPARATE FILE
;----------------------------------------------------------    
!source "assets.asm"

;----------------------------------------------------------
; MAIN PROGRAM
;----------------------------------------------------------
Launcher:
    sei
    cld
    ldx #$ff
    txs

    ; --- KERNAL SETUP ---
    jsr IOINIT

    ; --- FAST RAM SETUP WITHOUT CHECK
    lda #$00
    tay
.looprc:
    sta $0002,y
    sta $0200,y
    sta $0300,y
    iny
    bne .looprc
    ldx #$3c
    ldy #$03
    stx $b2
    sty $b3
    ldx #$00
    ldy #$a0
    jsr $fd8c
    ; --- END OF ABBR. RAM CHECK
    
    jsr RESTOR
    jsr CINT
    
    lda #$00
    jsr SETMSG
    cli

    ; --- VIDEO SETUP ---
    lda CIA2_PRA
    ora #$03
    sta CIA2_PRA
    lda #$14
    sta VIC_MEM_PTR

    ; --- COLORS ---
    lda #COL_BORDER
    sta VIC_BORDER
    lda #COL_BACKGROUND
    sta VIC_BG

    ; --- INIT ---
    lda #0
    sta SelectionIndex
    sta InputDelay
    sta TempColor

    jsr FlushBuffer
    jsr DrawBackground
    jsr DrawStaticText     
    jsr ResetTimer

MenuLoop:
    jsr WaitFrame
    jsr DrawMenu
    jsr CheckInput
    bcc .NoActivity

    jsr ResetTimer
    jmp .LoopEnd

.NoActivity:
    jsr ProcessTimer

.LoopEnd:
    jmp MenuLoop

;----------------------------------------------------------
; INPUT HANDLER
;----------------------------------------------------------
CheckInput:
    lda InputDelay
    beq .Ready
    dec InputDelay
    clc
    rts

.Ready:
    !if USE_PORT_1 = 1 {
        ; --- MODE 1: PORT 1 ONLY ---
        lda CIA1_PRB
        and #$1F
        sta zp_joy_state

        lda zp_joy_state
        and #1
        beq .ActUp
        lda zp_joy_state
        and #2
        beq .ActDown
        lda zp_joy_state
        and #16
        beq .ActFire
    } else {
        ; --- MODE 2: PORT 2 + KEYBOARD ---
        jsr SCNKEY
        jsr GETIN
        
        cmp #145
        beq .ActUp
        cmp #17
        beq .ActDown
        cmp #13
        beq .ActFire
        cmp #32
        beq .ActFire

        lda CIA1_PRA
        and #$1F
        sta zp_joy_state

        lda zp_joy_state
        and #1
        beq .ActUp
        lda zp_joy_state
        and #2
        beq .ActDown
        lda zp_joy_state
        and #16
        beq .ActFire
    }
    clc
    rts

.ActUp:
    lda SelectionIndex
    beq .Moved
    dec SelectionIndex
    jmp .Moved
.ActDown:
    lda SelectionIndex
    cmp #MAX_ITEM_IDX
    beq .Moved
    inc SelectionIndex
    jmp .Moved
.ActFire:

  .WaitButtonRelease:
      lda #%00010000
      bit $dc00
      beq .WaitButtonRelease
      bit $dc01
      beq .WaitButtonRelease

  .WaitNoKeyboard:
      ldy $c5
      cpy #$40
      bne .WaitNoKeyboard
  
    ldx SelectionIndex
    lda MenuBanks, x
    sta $FB
    jmp LaunchCart
    
.Moved:
    lda #10
    sta InputDelay
    sec
    rts

;----------------------------------------------------------
; SUBROUTINE: DrawStaticText
; Renders the 3 fixed strings defined in ASSETS
;----------------------------------------------------------
DrawStaticText:
    ; 1. INSTRUCTIONS (Line 6)
    lda #<TxtInstructions
    sta zp_text_ptr
    lda #>TxtInstructions
    sta zp_text_ptr + 1
    ldx #6                  ; Row 6
    lda #COL_INSTRUCTIONS
    jsr PrintCentered

    ; 2. GAME NAME (Line 9)
    lda #<TxtGameName
    sta zp_text_ptr
    lda #>TxtGameName
    sta zp_text_ptr + 1
    ldx #9                  ; Row 9
    lda #COL_GAMENAME
    jsr PrintCentered

    ; 3. COMMENT (Line 23)
    lda #<TxtComment
    sta zp_text_ptr
    lda #>TxtComment
    sta zp_text_ptr + 1
    ldx #23                 ; Row 23
    lda #COL_COMMENT
    jsr PrintCentered
    rts

;----------------------------------------------------------
; SUBROUTINE: DrawMenu
; Iterates items and calls PrintCentered
;----------------------------------------------------------
DrawMenu:
    lda #<MenuTextBase
    sta zp_text_ptr
    lda #>MenuTextBase
    sta zp_text_ptr + 1
    
    lda #0
    sta zp_item_idx

.ItemLoop:
    ; Calculate Row: 13 + (Item * 2)
    lda zp_item_idx
    asl                     ; * 2
    clc
    adc #13                 ; + 12 (Start Line)
    tax                     ; X = Row Number

    ; Determine Color
    ldy zp_item_idx
    cpy SelectionIndex
    beq .Highlight
    lda #COL_MENU_UNSEL
    bne .Render
.Highlight:
    lda #COL_MENU_SEL
.Render:
    ; Call the shared renderer
    ; Inputs: X=Row, A=Color, zp_text_ptr=String
    jsr PrintCentered

    ; Advance Text Pointer logic
    ; (This logic must calculate length again to advance ptr)
    ldy #0
.FindLen:
    lda (zp_text_ptr), y
    beq .FoundLen
    iny
    jmp .FindLen
.FoundLen:
    tya
    clc
    adc #1                  ; +1 for null
    adc zp_text_ptr
    sta zp_text_ptr
    bcc .PtrFix
    inc zp_text_ptr + 1
.PtrFix:

    inc zp_item_idx
    lda zp_item_idx
    cmp #ITEM_COUNT
    bne .ItemLoop
    rts

;----------------------------------------------------------
; SUBROUTINE: PrintCentered
; Inputs: 
;   zp_text_ptr = Address of string (null terminated)
;   X = Row Number (0-24)
;   A = Color Code
;----------------------------------------------------------
PrintCentered:
    sta TempColor           ; Save requested color
    stx zp_row_idx          ; Save row

    ; 1. Calculate Screen Address (Row * 40)
    ; $0400 + (Row * 40)
    lda #0
    sta zp_screen_ptr + 1
    lda zp_row_idx
    sta zp_screen_ptr       ; Start with Row
    
    ; Multiply by 8 (Shift Left 3)
    asl zp_screen_ptr
    rol zp_screen_ptr + 1
    asl zp_screen_ptr
    rol zp_screen_ptr + 1
    asl zp_screen_ptr
    rol zp_screen_ptr + 1   ; Now Row * 8
    
    ; Multiply by 5 ( (Row*8) * 4 + (Row*8) )
    ; Or... simpler approach: loop add 40.
    ; For menu speed, loop is fine (max 24 loops)
    
    lda #0
    sta zp_screen_ptr
    sta zp_screen_ptr + 1
    
    ldy zp_row_idx
    beq .CalcDone
    
    lda #<SCREEN_RAM
    sta zp_screen_ptr
    lda #>SCREEN_RAM
    sta zp_screen_ptr + 1

.CalcLoop:
    lda zp_screen_ptr
    clc
    adc #40
    sta zp_screen_ptr
    bcc .NoCarry
    inc zp_screen_ptr + 1
.NoCarry:
    dey
    bne .CalcLoop
    jmp .AddrReady

.CalcDone:
    lda #<SCREEN_RAM
    sta zp_screen_ptr
    lda #>SCREEN_RAM
    sta zp_screen_ptr + 1

.AddrReady:
    ; 2. Measure String Length (Max 38)
    ldy #0
.LenLoop:
    lda (zp_text_ptr), y
    beq .LenFound
    iny
    cpy #38                 ; Max width safety
    bne .LenLoop
.LenFound:
    sty zp_text_len

    ; 3. Calculate Centering Offset
    ; Offset = (40 - Length) / 2
    lda #40
    sec
    sbc zp_text_len
    lsr
    sta zp_center_off
    
    ; Apply Offset
    lda zp_screen_ptr
    clc
    adc zp_center_off
    sta zp_screen_ptr
    bcc .OffSkip
    inc zp_screen_ptr + 1
.OffSkip:

    ; 4. Setup Color Pointer
    lda zp_screen_ptr
    sta zp_color_ptr
    lda zp_screen_ptr + 1
    clc
    adc #$D4                ; $D800 - $0400
    sta zp_color_ptr + 1

    ; 5. Print Loop
    ldy #0
.PrintLoop:
    cpy zp_text_len
    beq .PrintDone
    lda (zp_text_ptr), y
    sta (zp_screen_ptr), y
    lda TempColor
    sta (zp_color_ptr), y
    iny
    jmp .PrintLoop
.PrintDone:
    rts

;----------------------------------------------------------
; UTILS & LAUNCHER
;----------------------------------------------------------
ProcessTimer:
    lda TimerLo
    sec
    sbc #1
    sta TimerLo
    lda TimerHi
    sbc #0
    sta TimerHi
    bcc .Timeout
    rts
.Timeout:
    ldx SelectionIndex
    lda MenuBanks,x
    sta $FB
    jmp LaunchCart

ResetTimer:
    lda #<TIMEOUT_FRAMES
    sta TimerLo
    lda #>TIMEOUT_FRAMES
    sta TimerHi
    rts

WaitFrame:
    bit VIC_CTRL1
    bpl *-3
    bit VIC_CTRL1
    bmi *-3
    rts

FlushBuffer:
    jsr GETIN
    cmp #0
    bne FlushBuffer
    rts

DrawBackground:
    ldx #0
.BgLoop:
    lda ScreenData, x
    sta SCREEN_RAM, x
    lda ColorData, x
    sta COLOR_RAM, x
    lda ScreenData+250, x
    sta SCREEN_RAM+250, x
    lda ColorData+250, x
    sta COLOR_RAM+250, x
    lda ScreenData+500, x
    sta SCREEN_RAM+500, x
    lda ColorData+500, x
    sta COLOR_RAM+500, x
    lda ScreenData+750, x
    sta SCREEN_RAM+750, x
    lda ColorData+750, x
    sta COLOR_RAM+750, x
    inx
    cpx #250
    bne .BgLoop
    rts

LaunchCart:
    sei
    ldx #EndTrampoline - TrampolineCode
.Copy:
    lda TrampolineCode, x
    sta TAPE_BUFFER, x
    dex
    bpl .Copy
    lda $FB
    jmp TAPE_BUFFER

TrampolineCode:
    sta MAGIC_LATCH
    jmp (RESET_VEC)
EndTrampoline:

!align $2000, $2000, $ff