import sys
import os

# ============================================
# Multidesk Project
# File Patcher for Oneload64 files
# Patches a .bin (NOT .crt) to a specific bank
# Full docs at www.dreisechzig.net/multidesk
# ============================================


# ==========================================
# CONFIGURATION SECTION
# ==========================================

# 1. The 8-byte signature to verify (Change this to your specific signature)
# Example: "STATMAT " (Oneload signature)
EXPECTED_SIGNATURE = b'\x53\x54\x41\x54\x4d\x41\x54\x20' 

# 2. The position X where the signature is located
SIGNATURE_OFFSET = 0x0000b 

# 3. List of positions to modify (Hexadecimal notation)
# These are examples. Replace these with your specific 5-digit hex offsets.
TARGET_POSITIONS = [
	(0x000cf, 0x00),
	(0x000ec, 0x09),
	(0x001c4, 0x08),
	(0x001e4, 0x00),
	(0x128ea, 0x08),
	(0x129b7, 0x09),
	(0x12a72, 0x00),
	(0x12b22, 0x0a),
	(0x12b71, 0x09)
	]

# ==========================================
# MAIN LOGIC
# ==========================================

def modify_binary_file(filename, add_value):
    """
    Validates signature, checks expected values at offsets, and modifies bytes.
    Aborts on value mismatch. Ends early on EOF.
    """
    
    # 1. Validate Input Number
    if not (0 <= add_value <= 118):
        print(f"Error: The number must be between 0 and 118. You entered {add_value}.")
        return

    # 2. Check if file exists
    if not os.path.isfile(filename):
        print(f"Error: File '{filename}' not found.")
        return

    file_size = os.path.getsize(filename)

    try:
        with open(filename, 'r+b') as f:
            
            # --- Signature Verification ---
            if file_size < SIGNATURE_OFFSET + 8:
                 print("Error: File is too small to contain the signature.")
                 return

            f.seek(SIGNATURE_OFFSET)
            current_signature = f.read(8)
            
            if current_signature != EXPECTED_SIGNATURE:
                print("ABORT: Signature mismatch.")
                print(f"Expected: {EXPECTED_SIGNATURE.hex().upper()}")
                print(f"Found:    {current_signature.hex().upper()}")
                return
            
            print(f"Signature verified successfully.")
            print("-" * 60)
            print(f"{'POSITION':<10} | {'EXPECTED':<10} | {'FOUND':<10} | {'NEW VAL':<10}")
            print("-" * 60)

            # --- Modification Loop ---
            for pos, expected_val in TARGET_POSITIONS:
                
                # Check for End of File (EOF)
                # We need to read 1 byte, so pos must be < file_size
                if pos >= file_size:
                    print(f"End of file reached before position 0x{pos:05X}. Ending early.")
                    break 

                # Move to the specific position
                f.seek(pos)
                
                # Read 1 byte
                byte_data = f.read(1)
                
                # Convert byte to integer
                actual_val = int.from_bytes(byte_data, byteorder='little')
                
                # --- VALIDATION STEP ---
                if actual_val != expected_val:
                    print(f"0x{pos:05X}    | {expected_val:<10} | {actual_val:<10} | ---")
                    print("-" * 60)
                    print(f"ABORT: Value mismatch at 0x{pos:05X}.")
                    print(f"Expected {expected_val} but found {actual_val}. No changes made to this byte.")
                    return # Stop the entire script immediately

                # --- CALCULATION STEP ---
                # Perform 8-bit addition (modulo 256)
                new_val = (actual_val + add_value) % 256
                
                # Move back to overwrite
                f.seek(pos)
                f.write(bytes([new_val]))
                
                # Output success for this byte
                print(f"0x{pos:05X}    | {expected_val:<10} | {actual_val:<10} | {new_val:<10}")

            print("-" * 60)
            print("Process finished.")

    except PermissionError:
        print(f"Error: Permission denied. Close the file in other programs.")
    except Exception as e:
        print(f"An unexpected error occurred: {e}")

if __name__ == "__main__":
    print("Binary File Patcher & Validator")
    
    if len(sys.argv) == 3:
        fname = sys.argv[1]
        try:
            num = int(sys.argv[2])
            modify_binary_file(fname, num)
        except ValueError:
            print("Error: The second argument must be an integer.")
    else:
        fname = input("Enter file name: ")
        try:
            num_input = input("Enter number to add (0-118): ")
            num = int(num_input)
            modify_binary_file(fname, num)
        except ValueError:
            print("Error: Please enter a valid integer.")